<?php
/* --------------------------------------------------------------
   SqlFilters.php 2021-08-11
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Filter;

use Doctrine\DBAL\Query\QueryBuilder;

/**
 * Class SqlFilters
 *
 * @package Gambio\Core\Filter
 */
abstract class SqlFilters extends LooseFilters
{
    /**
     * @return array
     */
    abstract public static function attributeColumnMapping(): array;
    
    
    /**
     * @return array
     */
    abstract public static function attributesForLikeOperation(): array;
    
    
    /**
     * @return array
     */
    abstract public static function attributesForNumericOperations(): array;
    
    
    /**
     * @param QueryBuilder $query
     * @param bool         $addCondition
     */
    public function applyToQuery(QueryBuilder $query, bool $addCondition = true): void
    {
        foreach ($this->filters as $filter) {
            $attribute = $filter->attribute();
            if (array_key_exists($attribute, static::attributeColumnMapping()) === false) {
                continue;
            }
            
            $operation = $filter->operation();
            $column    = static::attributeColumnMapping()[$attribute];
            $value     = $filter->value();
            $parameter = ':criteria_filter_' . str_replace('.', '_', $column);
            
            if ($operation === 'like' && in_array($attribute, static::attributesForLikeOperation(), true)) {
                $value      = str_replace(['%', '*'], ['\%', '%'], $value);
                $filterExpr = $query->expr()->like($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            } elseif ($operation === 'gt' && in_array($attribute, static::attributesForNumericOperations(), true)) {
                $filterExpr = $query->expr()->gt($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            } elseif ($operation === 'gte' && in_array($attribute, static::attributesForNumericOperations(), true)) {
                $filterExpr = $query->expr()->gte($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            } elseif ($operation === 'lt' && in_array($attribute, static::attributesForNumericOperations(), true)) {
                $filterExpr = $query->expr()->lt($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            } elseif ($operation === 'lte' && in_array($attribute, static::attributesForNumericOperations(), true)) {
                $filterExpr = $query->expr()->lte($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            } elseif ($operation === 'neq') {
                $filterExpr = $query->expr()->neq($column, $parameter);
                $query->andWhere($filterExpr)->setParameter($parameter, $value);
            } else {
                $filterExpr = $query->expr()->eq($column, $parameter);
                if ($addCondition) {
                    $query->andWhere($filterExpr);
                }
                
                $query->setParameter($parameter, $value);
            }
        }
    }
}